import { Injectable } from '@nestjs/common';
import {
  SensorData,
  EvaluationResult,
  RawSensorPayload,
  SENSOR_NAMES,
  SensorName,
} from '../fuzzy-core/fuzzy.types';
import { Observable, map, scan, filter } from 'rxjs';
import { MqttService } from '../mqtt-core/mqtt.service';
import { IncomingMessage } from 'http';
import { machineConfig } from '../fuzzy-core/fuzzy.engine.factory';
import { FuzzyService } from '../fuzzy-core/fuzzy.service';

/**
 * MaintenanceService handles the evaluation of sensor data from machines.
 * It subscribes to MQTT topics for live updates and processes incoming data
 * to provide evaluations based on fuzzy logic.
 */
@Injectable()
export class MaintenanceService {
  private latestEvaluations: EvaluationResult[] = [];

  constructor(
    private mqttService: MqttService,
    private fuzzyService: FuzzyService,
  ) {}

  /**
   * Returns the latest evaluation results.
   * @return {EvaluationResult[]} Array of the latest evaluation results.
   */
  getLatestEvaluations(): EvaluationResult[] {
    return this.latestEvaluations;
  }

  /**
   * Subscribes to MQTT topics for live updates and processes incoming data.
   * @param request Incoming HTTP request.
   * @return {Observable<EvaluationResult[]>} Observable of evaluation results.
   */
  getEvaluations(request: IncomingMessage): Observable<EvaluationResult[]> {
    return this.mqttService.getLiveUpdates('#', request).pipe(
      map(({ data }) => this.parseSensorData(data)),
      scan<SensorData | null, SensorData[]>((acc, incoming) => {
        if (!incoming) return acc;

        const idx = acc.findIndex((s) => s.machineId === incoming.machineId);
        if (idx !== -1) {
          acc[idx] = { ...acc[idx], ...incoming };
        } else {
          acc.push(incoming);
        }

        return acc;
      }, []),
      map((sensorData) => {
        const complete = sensorData.filter((s) => {
          const expectedSensors = machineConfig[s.machineId]?.sensors ?? [];
          return expectedSensors.every((sensor) => s[sensor] !== undefined);
        });

        const evaluations = complete.map((data) =>
          this.fuzzyService.evaluate(data),
        );

        const sorted = evaluations.sort((a, b) => b.score - a.score);
        this.latestEvaluations = sorted;

        return sorted;
      }),
      filter((evaluations) => evaluations.length > 0),
    );
  }

  /**
   * Parses raw sensor data from MQTT messages.
   * @param raw Raw sensor data string.
   * @returns Parsed partial SensorData, or null if invalid.
   */
  private parseSensorData(raw: string): Partial<SensorData> | null {
    try {
      const parsed: unknown = JSON.parse(raw);

      if (!this.isValidSensorPayload(parsed)) {
        console.warn('Invalid structure or types:', raw);
        return null;
      }

      const { machineId, sensorType, value } = parsed;
      const sensor = sensorType as SensorName;

      if (!SENSOR_NAMES.includes(sensor)) {
        console.warn(`Unsupported sensorType: ${sensor}`);
        return null;
      }

      return { machineId, [sensor]: value };
    } catch {
      console.warn('Failed to parse sensor data JSON:', raw);
      return null;
    }
  }

  /**
   * Validates the structure and types of the sensor payload.
   * @param input The input to validate.
   * @returns True if valid, false otherwise.
   */
  private isValidSensorPayload(input: unknown): input is RawSensorPayload {
    if (typeof input !== 'object' || input === null) {
      return false;
    }

    const obj = input as Record<string, unknown>;

    return (
      typeof obj.machineId === 'string' &&
      typeof obj.sensorType === 'string' &&
      typeof obj.value === 'number'
    );
  }
}
